#!/usr/bin/python3
# ==============================================================================================
# Serial design
# Really slow when reading. Think about using PyTTY alternative
# PySerial() class sets STOPBITS_ONE_POINT_FIVE as STOPBITS_TWO !
#    This leads to miss characters on read while setting it to STOPBITS_ONE will do the job !
# Using PosixPollSerial seems to be better on Rasp
# ==============================================================================================
class Serial:
    """RTTY_Serial.Serial Class: Print to the serial link"""
    import serial
    # ------------------------------------------------------------------------------------------
    def __init__(self,port="NONE",baud=50,size=serial.FIVEBITS,stop=serial.STOPBITS_ONE,parity=serial.PARITY_NONE,dbg=False):
        self._dbg     = dbg
        self._running = False
        self._exists  = False
        self._opened  = False
        self._ser     = None
        self._baud    = baud
        self._size    = size
        self._stop    = stop
        self._parity  = parity
        self._port    = self.porttst(port)
        self._timeout = 0.2
        self._read    = ""
   # ------------------------------------------------------------------------------------------
    def __repr__(self):
        return "port:'{}' baud:{} size:{} stop: {} parity:{} timeout:{}".format(self.port,self._baud,self._size,self._stop,self._parity)
    # ------------------------------------------------------------------------------------------
    @property
    def dbg(self):
        return self._dbg
    @dbg.setter
    def dbg(self,x):
        self._dbg=x
    # ------------------------------------------------------------------------------------------
    @property
    def opened(self):
        return self._opened
    # ------------------------------------------------------------------------------------------
    @property
    def running(self):
        return(self._running)
    # ------------------------------------------------------------------------------------------
    @property
    def exists(self):
        return self._exists
    # ------------------------------------------------------------------------------------------
    @property
    def in_waiting(self):
        return self._ser.in_waiting
    # ------------------------------------------------------------------------------------------
    @property
    def port(self):
        return self._port
    @port.setter
    def port(self,x):
        self._port=self.porttst(x)
    # ------------------------------------------------------------------------------------------
    @property
    def baud(self):
        return self._baud
    @baud.setter
    def baud(self,x):
        self._baud=x
    # ------------------------------------------------------------------------------------------
    @property
    def size(self):
        return self._size
    @size.setter
    def size(self,x):
        self._size=x
    # ------------------------------------------------------------------------------------------
    @property
    def stop(self):
        return self._stop
    @stop.setter
    def stop(self,x):
        self._stop=x
    # ------------------------------------------------------------------------------------------
    @property
    def parity(self):
        return self._parity
    @parity.setter
    def parity(self,x):
        self._parity=x
    # ------------------------------------------------------------------------------------------
    @property
    def timeout(self):
        return self._timeout
    @timeout.setter
    def timeout(self,x):
        self._timeout=x
    # ------------------------------------------------------------------------------------------
    def porttst(self,x):
        # --------------------------------------------------------------------------------------
        # Get system information
        # --------------------------------------------------------------------------------------
        if (x=="NONE"):
            import os, sys
            from sys import platform
            syst=platform.lower()
            if "win"     in syst:
                x="COM1"
            if "linux" or "freebsd" in syst:
                if   (os.path.exists("/dev/serial0")): x="/dev/serial0"
                elif (os.path.exists("/dev/ttyS0")):   x="/dev/ttyS0"
                else: raise ValueError("Serial device not found")
        # --------------------------------------------------------------------------------------
        # Try to open and default to classic ASCII output if no serial is there
        # --------------------------------------------------------------------------------------
        try:
            self._exists=True
            self._ser=self.serial.Serial(x,self._baud)
        except Exception as e:
            self._exists=False
            os.system('stty sane')
            sys.tracebacklimit = None
            raise e
        self._ser.reset_output_buffer()
        self._ser.close()
        return(x)
    # ------------------------------------------------------------------------------------------
    def Close(self):
        if (self._opened):
            try:
                self._ser.close()
                self._running=False
                self._opened=False
            except:
                pass
    # ------------------------------------------------------------------------------------------
    def Open(self):
        self._opened=False
        self._running=False
        if (self._exists):
            try:	
                # ------------------------------------------------------------------------------
                # Alternates method for Posix, firts don't works, second is also slow
                # ------------------------------------------------------------------------------
                #self._ser=self.serial.serial_for_url("alt://"+self._port+"?class=PosixPollSerial")
                #self._ser=self.serial.serial_for_url("alt://"+self._port+"?class=VTIMESerial")
                #self._ser.close()
                # ------------------------------------------------------------------------------
                self._ser=self.serial.Serial()
                # ------------------------------------------------------------------------------
                self._ser.port     = self._port
                self._ser.baudrate = self._baud
                self._ser.bytesize = self._size
                self._ser.parity   = self._parity
                self._ser.stopbits = self._stop
                self._ser.xonxoff  = False
                self._ser.rtscts   = False
                self._ser.dsrdtr   = False
                self._ser.timeout           = self._timeout
                self._ser.write_timeout     = None
                self._ser.inter_byte_timeout= None
                self._ser.open()
                if (self._ser.is_open): self._opened=True
            except Exception as e: 
#               raise
                pass
    # ------------------------------------------------------------------------------------------
    def Read(self,size=1):				# Outputs a <string> type
        data=rb""
        if (self._opened):
            try:
                self._running = True
                data = self._ser.read(size)
                self._running = False
            except:
                pass
        return(data.decode('ascii'))
    # ------------------------------------------------------------------------------------------
    def Print(self,data):				# Input <string> type
        if (self._opened):
            try:
                self._running = True
                if (self._dbg): print("INFO:  SERIAL PRINT START")
                self._ser.write(data.encode('ascii'))
                if (self._dbg): print("INFO:  SERIAL PRINT END")
                self.Wait()
                if (self._dbg): print("INFO:  SERIAL PRINT SENT ALL")
                self._running = False
            except Exception as e:
                if (self._dbg): print("ERROR: SERIAL PRINT NOT STARTED ({0})".format(e))
                pass
    # ------------------------------------------------------------------------------------------
    def Abort(self):
       if (self._running):
            self._running = False
            import time
            self._ser.cancel_write()
            self._ser.reset_output_buffer()
            time.sleep(0.3)
            if (self._dbg): print("INFO:  SERIAL PRINT KILLED")
 # ------------------------------------------------------------------------------------------        
    def KillConsole(self):      # Killing from direct console launch
        import os
        self.Abort()
        os.system('stty sane')            
    # ------------------------------------------------------------------------------------------
    def Wait(self):
       while (self._opened) and (self._ser.out_waiting!=0):
           continue
    # ------------------------------------------------------------------------------------------
    def Test(self):
        import binascii
        try:
            self.Open()
            print("TEST:  SERIAL LINK RX (10 chars in "+str(self._timeout)+"s max.)")
            data = self.Read(10)
            if (data):
                print("->     "+":".join("{:02x}".format(ord(b)) for b in data))
                from pprint import pprint
    #           pprint(data)
            else:
                if (self._dbg): print("TEST:  SERIAL LINK RX TIMEOUT")

            print("TEST:  SERIAL LINK TX")
            for i in range(4):
                if (self._size!=5):
                    self.Print("THEQUICKBROWNFOXJUMPSOVERTHELAZYDOG0123456789 ()?!/.'+-=;&%\r\n")
                else:
                    self.Print("\x1f\x10\x14\x01\x17\x07\x06\x0e\x0f\x19\x0a\x18\x13\x0c\x0d\x18\x1d\x0b"+
                            "\x07\x1c\x16\x05\x18\x1e\x01\x0a\x10\x14\x01\x12\x03\x11\x15\x09\x18\x1a"+
                            "\x1b\x16\x17\x13\x01\x0a\x10\x15\x07\x06\x18\x1f\x04\x1b\x0f\x12\x19\x1f"+
                            "\x04\x1b\x1d\x1c\x05\x11\x03\x1e\x1f\x04\x04\x1b\x1a\x02\x08")
            self.Close()
        except:
            print("TEST:  SERIAL LINK")
            print("-> Not passed")
            pass            
# ==============================================================================================
if __name__ == '__main__':
    import sys
    import signal
    from inspect import getdoc
    def signal_term_handler(signal, frame):
        this.KillConsole()
        sys.exit(0)
    signal.signal(signal.SIGINT,  signal_term_handler)
    signal.signal(signal.SIGTERM, signal_term_handler)
    print(getdoc(Serial))
    this = Serial()
    this.dbg = True
    this.timeout = 20
    this.Test()
# ==============================================================================================
