#!/usr/bin/python3
# =============================================================================================
# Audio design 
# ==============================================================================================
class Audio:
    """RTTY_Audio.Audio Class: Process a local .wav sound file"""
    # ------------------------------------------------------------------------------------------
    def __init__(self,path="./sound/",dbg=False):
        self._dbg = dbg
        self._running = False
        self._exists = True
        self.path = path
        self.player = "aplay"
        self.pid = None
        self.p  = 0
    # ------------------------------------------------------------------------------------------
    def __repr__(self):
        return "player:'{}' path:'{}'".format(self.player, self._path)
    # ------------------------------------------------------------------------------------------
    @property
    def dbg(self):
        return self._dbg
    @dbg.setter
    def dbg(self, x):
        self._dbg = x
    # ------------------------------------------------------------------------------------------
    @property
    def running(self):
        return self._running
    # ------------------------------------------------------------------------------------------
    @property
    def exists(self):
        return self._exists
    # ------------------------------------------------------------------------------------------
    @property
    def path(self):
        return self._path
    @path.setter
    def path(self, x):
        import os, sys
        self._exists = True
        if (os.path.exists(x)):
            self._path = x
        else:
            self._path = ""
            self._exists = False
            sys.tracebacklimit = None
            raise ValueError("Audio sound path '"+x+"' not found")
    # ------------------------------------------------------------------------------------------
    @property
    def player(self):
        return self._player
    @player.setter
    def player(self, x):
        import os,sys
        self._exists = True
        try:
            res = os.system(x + " -h >/dev/null 2>&1")
        except Exception as e:
            self._player = ""
            self._exists = False
            sys.tracebacklimit = None
            raise ValueError("Audio sound player '"+x+"' not found")
        self._player = x
        if (res != 0):
            self._player = ""
            self._exists = False
            sys.tracebacklimit = None
            raise ValueError("Audio sound player '"+x+"' not found")
    # -------------------------------------------------------------------------------------------        
    def Play(self, name):        
        import os
        if (self._exists) and (os.path.exists(self._path+name)):
            try:
                import os, subprocess
                self._running = True
                devnull = open('/dev/null', 'w')
                self.p = subprocess.Popen([self._player, "-q", self._path+name], preexec_fn=os.setsid, stdout=devnull, shell=False)
                self.pid = self.p.pid
                if (self._dbg): print("INFO:  AUDIO  PROCESS STARTED")
                self.Wait()
                self._running = False
                if (self._dbg): print("INFO:  AUDIO  PROCESS ENDED")
            except Exception as e:
                self._running = False
                if (self._dbg): print("ERROR: AUDIO  PROCESS NOT STARTED ({0})".format(e))
                pass
    # ------------------------------------------------------------------------------------------        
    def Abort(self):            # Process to be killed has been detached from console (os.setsid)
        import signal, os       # To avoid killing father process
        if (self._running):
            self._running = False
            try:
                os.killpg(os.getpgid(self.pid), signal.SIGTERM)
                self.Wait()
                if (self._dbg): print("INFO:  AUDIO  PROCESS KILLED")
            except Exception as e:
                if (self._dbg): print("ERROR: AUDIO  PROCESS NOT KILLED ({0})".format(e))
                pass
    # ------------------------------------------------------------------------------------------        
    def KillConsole(self):      # Killing from direct console launch
        import os
        if (self._running):
            os.killpg(os.getpgid(self.pid), signal.SIGTERM)
            os.system('stty sane')
    # ------------------------------------------------------------------------------------------
    def Wait(self): 
        self.p.wait()
    # ------------------------------------------------------------------------------------------
    def Test(self):
        print("TEST:  AUDIO  PLAYER")
        try:
            self.Play("Test.wav") 
        except:
            print("-> Not passed")
            pass
# ==============================================================================================
if __name__ == '__main__':
    import sys
    import signal
    from inspect import getdoc
    def signal_term_handler(signal, frame):
        this.KillConsole()
        sys.exit(0)
    signal.signal(signal.SIGINT,  signal_term_handler)
    signal.signal(signal.SIGTERM, signal_term_handler)
    print(getdoc(Audio))
    this = Audio()
    this.dbg = True
    this.Test()
# ==============================================================================================
